<?php
declare(strict_types=1);

final class IncidentsController {
  public static function list(): void {
    $u = require_login();
    $pdo = db();
    if ($u['role'] === 'admin') {
      $rows = $pdo->query('SELECT i.id, i.title, i.status, i.started_at, i.ended_at, i.location_text, i.created_at
        FROM incident i ORDER BY i.created_at DESC LIMIT 200')->fetchAll();
    } else {
      $stmt = $pdo->prepare('SELECT i.id, i.title, i.status, i.started_at, i.ended_at, i.location_text, i.created_at
        FROM incident i
        JOIN incident_org io ON io.incident_id = i.id
        WHERE io.org_id = :oid
        ORDER BY i.created_at DESC LIMIT 200');
      $stmt->execute([':oid'=>$u['org_id']]);
      $rows = $stmt->fetchAll();
    }
    api_json(['ok'=>true, 'incidents'=>$rows]);
  }

  public static function create(): void {
    $u = require_role(['admin','el']);
    $in = json_input();
    $title = trim((string)($in['title'] ?? ''));
    if ($title === '') api_error(400, 'title required');

    $location = trim((string)($in['location_text'] ?? ''));
    $orgIds = $in['org_ids'] ?? [];
    if (!is_array($orgIds) || count($orgIds) === 0) {
      // default: creator org only
      $orgIds = [$u['org_id']];
    }

    $id = null;
    db_tx(function(PDO $pdo) use ($title, $location, $orgIds, $u, &$id) {
      $stmt = $pdo->prepare('INSERT INTO incident(id, title, location_text, status, started_at, created_by_user_id)
        VALUES (gen_random_uuid(), :t, :loc, :st, now(), :uid)
        RETURNING id');
      $stmt->execute([':t'=>$title, ':loc'=>$location, ':st'=>'active', ':uid'=>$u['id']]);
      $id = (string)$stmt->fetchColumn();

      $io = $pdo->prepare('INSERT INTO incident_org(incident_id, org_id, role_in_incident, created_at)
        VALUES (:iid, :oid, :r, now())');
      foreach ($orgIds as $oid) {
        if (!is_string($oid) || !preg_match('/^[a-f0-9\-]{36}$/', $oid)) continue;
        $io->execute([':iid'=>$id, ':oid'=>$oid, ':r'=>'participant']);
      }

      $pdo->prepare('INSERT INTO audit_log(id, incident_id, actor_user_id, action, entity_type, entity_id, meta_json)
        VALUES (gen_random_uuid(), :iid, :uid, :act, :et, :eid, :m::jsonb)')
        ->execute([':iid'=>$id, ':uid'=>$u['id'], ':act'=>'incident.create', ':et'=>'incident', ':eid'=>$id, ':m'=>json_encode(['title'=>$title])]);
    });

    EventsController::emitIncident($id, null, 'all', 'incident.created', ['incident_id'=>$id]);
    api_json(['ok'=>true, 'id'=>$id], 201);
  }

  public static function get(string $id): void {
    $u = require_login();
    $pdo = db();
    $stmt = $pdo->prepare('SELECT id, title, status, started_at, ended_at, location_text, settings_json, created_at FROM incident WHERE id=:id');
    $stmt->execute([':id'=>$id]);
    $inc = $stmt->fetch();
    if (!$inc) api_error(404, 'incident not found');

    if ($u['role'] !== 'admin') {
      $chk = $pdo->prepare('SELECT 1 FROM incident_org WHERE incident_id=:iid AND org_id=:oid');
      $chk->execute([':iid'=>$id, ':oid'=>$u['org_id']]);
      if (!$chk->fetchColumn()) api_error(403, 'not a participant of this incident');
    }

    $orgs = $pdo->prepare('SELECT o.id, o.name, o.short_name, o.color FROM org o JOIN incident_org io ON io.org_id=o.id WHERE io.incident_id=:iid ORDER BY o.name');
    $orgs->execute([':iid'=>$id]);

    $inc['settings'] = json_decode($inc['settings_json'] ?? '{}', true);
    unset($inc['settings_json']);

    api_json(['ok'=>true, 'incident'=>$inc, 'orgs'=>$orgs->fetchAll()]);
  }

  public static function update(string $id): void {
    $u = require_role(['admin','el']);
    $in = json_input();

    $fields = [];
    $params = [':id'=>$id];
    if (isset($in['title'])) { $fields[] = 'title=:t'; $params[':t']=trim((string)$in['title']); }
    if (isset($in['location_text'])) { $fields[] = 'location_text=:loc'; $params[':loc']=trim((string)$in['location_text']); }
    if (isset($in['status'])) { $fields[]='status=:st'; $params[':st']=(string)$in['status']; }
    if (isset($in['settings']) && is_array($in['settings'])) { $fields[]='settings_json=:sj::jsonb'; $params[':sj']=json_encode($in['settings'], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); }

    if (!$fields) api_error(400, 'Nothing to update');

    $sql = 'UPDATE incident SET ' . implode(',', $fields) . ', updated_at=now() WHERE id=:id';
    db()->prepare($sql)->execute($params);

    db()->prepare('INSERT INTO audit_log(id, incident_id, actor_user_id, action, entity_type, entity_id, meta_json)
      VALUES (gen_random_uuid(), :iid, :uid, :act, :et, :eid, :m::jsonb)')
      ->execute([':iid'=>$id, ':uid'=>$u['id'], ':act'=>'incident.update', ':et'=>'incident', ':eid'=>$id, ':m'=>json_encode(['fields'=>array_keys($in)])]);

    EventsController::emitIncident($id, null, 'all', 'incident.updated', ['incident_id'=>$id]);
    api_json(['ok'=>true]);
  }
}
